---
theme: seriph
title: 基于 LeaferJS 封装的底图渲染与交互源码解析
info: |
  本演讲介绍如何在项目中使用 `mark/` 模块与 `leafer-mark-core/index.ts`，以及该模块通过封装 LeaferJS 的 `class LeaferAnnotate` 的核心设计、插件体系与关键方法。
class: text-center 
drawings:
  persist: false
transition: slide-left
mdc: true
defaults:
  layout: scroll
---

# 基于 LeaferJS 封装的底图渲染与交互源码解析



---

## 目录

1. 使用场景与目标
2. 快速上手（组件与 Hook）
3. 依赖关系：页面 → hook → LeaferAnnotate → LeaferJS
4. 运行时能力（API 与事件）
5. 源码解读：`LeaferAnnotate` 架构
6. 插件体系与交互模式
7. 数据流与落库
8. 扩展点与最佳实践

---

## 1. 使用场景与目标

- 标注 PDF 页面或图片底图上的题目/区域
- 支持拖拽、复制、吸附、标尺、对齐辅助线
- 统一封装：在 Vue 组件中复用一个 Leafer 标注实例

目标：简单接入、稳定交互、数据可序列化落库

---



## 2. 依赖关系
依赖链：LeaferJS ← LeaferAnnotate ← Hook ← 页面

- **LeaferJS（底层引擎）**: 负责 Canvas 渲染、事件与几何计算
- **LeaferAnnotate（业务实例）**: 基于 LeaferJS 封装标注能力与插件体系
- **Hook（useLeaferAnnotateSingleton）**: 管理单例、暴露页面可用的方法
- **页面（Vue 组件）**: 仅通过 Hook 与标注实例交互


页面不直接调用 Leafer/实例细节；Hook 只做编排与生命周期；实例聚合业务能力；底层交由 Leafer 处理


---

## 3. 快速使用
只需要在页面中引入 hook `useLeaferAnnotate`

要点：
- 调用 `createApp` 生成LeaferAnnotate实例，绑定到页面中
- 调用 `loadData(pageUrl, marks)` 加载底图与标注
- 调用 `loadPoints(pointsData)` 加载学员的书写笔迹


```ts
import { onMounted } from 'vue'
import { useLeaferAnnotateSingleton } from '@/mark/leafer-mark-core/useLeaferAnnotate'

const { createApp, loadData, loadPoints } = useLeaferAnnotateSingleton()

onMounted(async () => {
  // 初始化leafer canvas容器
  await createApp({ view: 'leafer-container' })
  // 加载底图与mark
  await loadData('https://example.com/page.png', [])
  // 加载学员笔迹
  loadPoints(pointsData)
})
```

---



## 4. LeaferAnnotate 主流程：init

入口在 Hook 的 `createApp`，最终委托到 `createLeaferAnnotate` 与 `LeaferAnnotate.init` 完成实例、画布与插件初始化：

```ts
export const createLeaferAnnotate = async (
  config: LeaferAnnotateConfig
): Promise<{
  getInstance: () => LeaferAnnotate | null
  destroy: () => Promise<void>
}> => {
  let instance: LeaferAnnotate | null = new LeaferAnnotate(cloneDeep(config))
  await instance.init()
```

`init` 核心步骤：创建 `App`、创建并挂载 `Frame`、启用插件、设置默认模式。

```ts
async init(): Promise<void> {
  let { view } = this.config

  // 初始化画布，没有设置宽高，默认使用父元素的宽高
  this.app = new App({
    view: view,
    ...DEFAULT_LEAFER_CONFIG
  })

  // // 设置图片和标记
  this.pageFrame = new Frame({
    id: 'pageFrame'
  })
  this.app.tree?.add(this.pageFrame)

  // 启用插件
  this.bindPlugins()

  // 默认为view模式
  this.changeMode('view')
}
```

要点：
- **App 创建**：传入 `view` 绑定 DOM 容器，合并默认渲染配置。
- **Frame 容器**：作为页面图层承载底图与所有标注矩形。
- **插件启用**：吸附、拖拽、复制、矩形工具、标尺、对齐辅助线一次性装配。
- **模式**：默认 `view`，可缩放、移动并可编辑元素；`edit` 则限制交互。

插件与模式的关键调用：

```ts
private bindPlugins(): void {
  this.snap = new Snap(this.app, {
    ...DEFAULT_SNAP_CONFIG,
    parentContainer: this.pageFrame
  })
  this.ruler = new Ruler(this.app, DEFAULT_RULER_CONFIG)

  // 精确控制元素，x,y,width,height在1px单位步进变化,去除小数点
  this.adsorptionBinding = new AdsorptionBinding()
  this.adsorptionBinding.install(this.app)
  // 在画布上拖拽图元
  this.dropBinding = new DropBinding()
  this.dropBinding.install(this)
  // 按住ctrl复制矩形
  this.copyRectBinding = new CopyRectBinding()
  this.copyRectBinding.install(this, DEFAULT_RECT_CONFIG.className)
  // 创建矩形工具
  this.createRectBinding = new CreateRectBinding()
  this.createRectBinding.install(this)
```

```ts
public changeMode(mode: 'view' | 'edit'): void {
  this.mode = mode
  this.pageFrame.cursor = 'crosshair'
  let interaction = this.app.tree?.interaction
  configureInteractionMode(interaction, mode)

  if (this.mode === 'view') {
    this.pageFrame.hitChildren = true
  } else if (this.mode === 'edit') {
    this.pageFrame.hitChildren = false
  }
}
```

---

## 4. LeaferAnnotate 主流程：loadData

`loadData` 负责清理旧内容、加载底图图片、缩放视口，并把接口标注转为矩形加入图层：

```ts
public async loadData(pageUrl: string, marks: IMark[]): Promise<void> {
  if (this.pageFrame) {
    this.pageFrame.clear()
    Resource.destroy()
  }
  // 获取底图，这是整个画布的核心
  let { url, width, height } = await loadImage(pageUrl, 'bg.png')

  this.pageFrame.width = width
  this.pageFrame.height = height

  this.app.tree?.zoom('fit-width', [12 + 20, 12, 12, 12 + 20])
  this.pageFrame.add(new Image({ id: 'bg-image', url: url, width: width, height: height }))

  // 初始化标注
  this.initMarks(marks)
}
```

要点：
- **清理**：`clear` 与 `Resource.destroy` 确保纹理与树干净。
- **底图**：异步 `loadImage` 拿到真实尺寸，设置 `Frame` 宽高后再添加背景 `Image`。
- **视口**：按页面宽度自适应缩放，留出标尺与外边距。
- **标注**：将服务端数据映射为 `Rect`，逐个加入 `Frame` 成为可交互图元。

---

## 4. LeaferAnnotate 主流程：loadMark（标注挂载）

标注转换与挂载：

```ts
private initMarks(marks: IMark[]): void {
  if (!Array.isArray(marks) || !this.pageFrame) return
  const layer = this.pageFrame
  const rects = processMarksToRects(marks)
  rects.forEach(rect => {
    layer.add(rect)
  })
}
```

要点：
- **数据映射**：`processMarksToRects` 将接口标注映射为可交互 `Rect`。
- **统一挂载**：逐个添加到 `Frame`，与底图处于同一画布树以便交互与渲染。

---
