# 基于 LeaferJS 的底图渲染与交互源码解析

这是一个基于 [Slidev](https://sli.dev/) 构建的技术演示文档项目，详细解析了基于 LeaferJS 封装的底图渲染与交互系统的实现原理。

## 📖 项目介绍

本项目深入介绍了如何在项目中使用 `mark/` 模块与 `leafer-mark-core/index.ts`，以及该模块通过封装 LeaferJS 的 `class LeaferAnnotate` 的核心设计、插件体系与关键方法。

### 核心内容

- **架构概览**：LeaferJS → LeaferAnnotate → Hook → Vue组件的完整技术栈
- **实现难点**：单位转换系统（像素与物理单位转换）
- **出血处理**：PDF印刷中的出血机制与实现

## 🏗️ 技术架构

### 依赖关系

```
LeaferJS (底层引擎) ← LeaferAnnotate (业务实例) ← Hook (单例管理) ← Vue组件 (页面交互)
```

- **LeaferJS**：负责 Canvas 渲染、事件和交互的底层引擎
- **LeaferAnnotate**：基于 LeaferJS 封装的业务逻辑实例
- **Hook**：通过 `useLeaferAnnotateSingleton` 管理单例，暴露页面可用方法
- **Vue组件**：仅通过 Hook 与标注实例交互，保持架构清晰

### 核心特性

- ✅ 基于 Canvas 的高性能渲染
- ✅ 支持标注的创建、选择和交互
- ✅ 完整的单位转换系统（px/mm/in）
- ✅ PDF出血处理机制
- ✅ Vue3 Hook 集成方案
- ✅ 单例模式管理

## 🚀 快速开始

### 环境要求

- Node.js 16+
- pnpm (推荐) 或 npm

### 安装依赖

```bash
pnpm install
```

### 开发环境

启动开发服务器，自动打开浏览器：

```bash
pnpm dev
```

访问 <http://localhost:3030> 查看演示。

### 构建生产版本

```bash
pnpm build
```

### 导出文档

导出为 PDF/PPT 格式：

```bash
pnpm export
```

## 📁 项目结构

```
bbb-canvas-docs/
├── slides.md              # 主演示文稿
├── pages/                 # 分页内容
│   ├── 架构概览.md        # 技术架构与依赖关系
│   ├── 实现难点.md        # 单位转换系统详解
│   └── 出血.md            # PDF出血处理机制
├── package.json           # 项目配置
├── vercel.json           # Vercel 部署配置
└── netlify.toml          # Netlify 部署配置
```

## 💡 核心功能示例

### LeaferAnnotate 使用

```typescript
import { createLeaferAnnotate } from "../mark/leafer-mark-core/index";

// 创建实例
const { getInstance, destroy } = await createLeaferAnnotate({
  view: document.getElementById("annotate-view"),
});

// 获取实例并加载数据
const leaferAnnotate = getInstance();
await leaferAnnotate?.loadData("https://example.com/page.png", [
  {
    id: "mark-1",
    x: 120,
    y: 80,
    width: 160,
    height: 60,
    questionID: 1001,
    color: "#FF6B6B",
  },
]);

// 交互操作
leaferAnnotate?.changeMode("view");
leaferAnnotate?.selectMark("mark-1");
leaferAnnotate?.setActiveQuestionID(1001);
leaferAnnotate?.resetView();
```

### Vue Hook 集成

```vue
<script setup lang="ts">
import { onMounted } from "vue";
import { useLeaferAnnotateSingleton } from "../mark/leafer-mark-core/useLeaferAnnotate";

const { createApp, loadData, selectMark, resetView } =
  useLeaferAnnotateSingleton();

onMounted(async () => {
  // 初始化
  await createApp({ view: "leafer-container" });

  // 加载数据
  await loadData("https://example.com/page.png", marks);
  selectMark("mark-1");
});
</script>
```

## 🔧 单位转换系统

项目实现了完整的单位转换体系，支持：

- **逻辑单位**：像素 (px)
- **物理单位**：毫米 (mm)、英寸 (in)
- **转换公式**：
  ```javascript
  像素数量 = 物理尺寸(英寸) × PPI
  物理尺寸(毫米) = 像素数量 ÷ PPI × 25.4
  ```

### 转换工具

```javascript
// 像素转毫米
export function processPxToMmPoint(point, ppi = 300, mm = 25.4) {
  return {
    x: (point.x / ppi) * mm,
    y: (point.y / ppi) * mm,
  };
}

// 毫米转像素
export function processMmToPxPoint(point, ppi = 300, mm = 25.4) {
  return {
    x: (point.x * ppi) / mm,
    y: (point.y * ppi) / mm,
  };
}
```

## 📄 PDF出血处理

实现了印刷标准的出血处理机制：

- **标准出血值**：3mm (约0.125英寸)
- **处理公式**：
  ```javascript
  const offsetX = bleed * (dpi / 25.4);
  const offsetY = bleed * (dpi / 25.4);
  ```

## 🎨 主题配置

- **主题**：seriph
- **绘图持久化**：关闭
- **MDC语法**：启用
- **布局**：滚动

## 🚀 部署

项目已配置自动部署支持：

### Vercel 部署

```json
{
  "rewrites": [{ "source": "/(.*)", "destination": "/index.html" }]
}
```

### Netlify 部署

```toml
[build]
  publish = "dist"
  command = "npm run build"
```

## 📚 相关资源

- [LeaferJS 官方文档](https://www.leaferjs.com/)
- [Slidev 官方文档](https://sli.dev/)
- [Vue 3 文档](https://vuejs.org/)

## 📄 许可证

本项目仅供学习和参考使用。

---

