# createLeaferAnnotate说明文档

创建 Leafer 标注实例的工厂函数。

#### 语法

```typescript
createLeaferAnnotate(config: LeaferAnnotateConfig): Promise<{
  getInstance: () => ILeaferAnnotate | null
  destroy: () => Promise<void>
}>
```

#### 参数

| 参数   | 类型                 | 必填 | 说明     |
| ------ | -------------------- | ---- | -------- |
| config | LeaferAnnotateConfig | 是   | 配置对象 |

#### LeaferAnnotateConfig 配置项

| 属性            | 类型                   | 必填 | 说明               |
| --------------- | ---------------------- | ---- | ------------------ |
| view            | string                 | 是   | 挂载的 DOM 元素 ID |
| pageUrl         | string                 | 是   | 背景图片的 URL     |
| marks           | IMark[]                | 是   | 标注数据数组       |
| onElementAdd    | (element: IUI) => void | 否   | 元素添加回调函数   |
| onElementSelect | (element: IUI) => void | 否   | 元素选择回调函数   |

#### IMark 标注数据结构

```typescript
interface IMark {
  top: IMarkPoint // 标注矩形左上角坐标
  bottom: IMarkPoint // 标注矩形右下角坐标
  id: string // 标注唯一标识
  PaperType?: string // 试卷类型（可选）
  questionID?: string | number // 题目ID（可选）
  childQuestionID?: string | number // 子题目ID（可选）
  [key: string]: any // 其他自定义属性
}

interface IMarkPoint {
  x: number // X 坐标
  y: number // Y 坐标
}
```

#### 返回值

返回一个包含以下方法的对象：

- `getInstance()`: 获取标注实例
- `destroy()`: 销毁标注实例

## 使用示例

### 基础用法

```typescript
import { createLeaferAnnotate } from './leafer-mark-core/index'
import type { ILeaferAnnotate } from './leafer-mark-core/index'

// 创建标注实例
const manager = await createLeaferAnnotate({
  view: 'leafer-container', // DOM 容器 ID
  pageUrl: 'https://example.com/image.png', // 背景图片 URL
  marks: [
    // 初始标注数据
    {
      id: 'mark-1',
      top: { x: 100, y: 100 },
      bottom: { x: 200, y: 150 },
      PaperType: 'math',
      questionID: 'q1'
    }
  ],
  onElementSelect: element => {
    console.log('选中元素:', element)
  },
  onElementAdd: element => {
    console.log('添加元素:', element)
  }
})

// 获取实例
const instance = manager.getInstance()
```

### 在 Vue 组件中使用

```vue
<template>
  <div class="w-full h-full bg-white leafer-container" id="leafer-container"></div>
</template>

<script setup lang="ts">
import { useBasic } from '@/hooks/useBasic'
import { createLeaferAnnotate } from './leafer-mark-core/index'
import type { ILeaferAnnotate } from './leafer-mark-core/index'

const { setBreadcrumb } = useBasic()
let instance = shallowRef<ILeaferAnnotate | null>(null)
let manager = shallowRef<{
  getInstance: () => ILeaferAnnotate | null
  destroy: () => Promise<void>
}>()

onMounted(async () => {
  setBreadcrumb([])
  await nextTick()

  // 创建标注实例
  manager.value = await createLeaferAnnotate({
    view: 'leafer-container',
    pageUrl: 'http://example.com/paper.png',
    marks: [],
    onElementSelect: element => {
      console.log('选中元素:', element)
    },
    onElementAdd: element => {
      console.log('添加元素:', element)
    }
  })

  instance.value = manager.value?.getInstance()
})

// 销毁实例
const setDestroy = async () => {
  if (manager.value) {
    try {
      await manager.value.destroy()
      manager.value = undefined
      instance.value = null

      await nextTick()

      const container = document.querySelector('#leafer-container')
      if (container) {
        container.innerHTML = ''
      }
    } catch (error) {
      console.error('销毁实例时出错:', error)
    }
  }
}

onUnmounted(() => {
  setDestroy()
})
</script>
```

## 实例方法

### resetView()

重置视图到适合宽度显示。

```typescript
instance.resetView()
```

### delElement(id: string)

根据 ID 删除指定元素。

```typescript
instance.delElement('mark-1')
```

### destroy()

销毁标注实例，清理所有资源。

```typescript
await instance.destroy()
```

## 注意事项

1. **容器要求**: 确保指定的 DOM 容器存在且有足够的空间显示标注
2. **图片加载**: 背景图片需要支持跨域访问
3. **资源清理**: 组件卸载时必须调用 `destroy()` 方法清理资源
4. **坐标系统**: 标注坐标基于图片的像素坐标系统
5. **交互模式**: 默认支持两种模式
   - `view`: 可缩放、移动页面，可编辑元素
   - `edit`: 不可缩放、移动，不可编辑元素

## 功能特性

- ✅ 图片标注矩形创建和编辑
- ✅ 标注数据的增删改查
- ✅ 视图缩放和移动
- ✅ 元素吸附和对齐
- ✅ 标尺显示
- ✅ 复制粘贴功能
- ✅ 拖拽操作
- ✅ 事件回调支持
