# Leafer Hook 使用说明

## 概述

`useLeaferAnnotate.ts` 提供了两个主要的 Vue 3 Composition API Hook，用于管理 Leafer 标注实例和限制配置：

- `useLeaferAnnotateSingleton()` - 单例标注实例管理
- `useLeaferLimit()` - 宽高限制与锁定状态管理

## useLeaferAnnotateSingleton

### 功能特性

- **单例模式**：跨多个子组件共享同一 Leafer 标注实例
- **引用计数**：自动管理实例生命周期，最后一个使用者卸载时自动销毁
- **事件总线**：基于 `@vueuse/core` 的多组件事件通信
- **响应式数据**：提供响应式的标记列表

### API 接口

#### createApp(config: LeaferAnnotateConfig)

创建标注实例并自动配置事件总线。

```typescript
const { createApp } = useLeaferAnnotateSingleton()

await createApp({
  container: '#container',
  width: 800,
  height: 600
})
```

#### loadData(pageUrl: string, marks: IMark[])

加载底图及标注数据。

```typescript
const { loadData } = useLeaferAnnotateSingleton()

await loadData('/path/to/image.jpg', [{ id: '1', x: 100, y: 100, width: 50, height: 50, type: 'rect' }])
```

#### getInstance()

获取底层 Leafer 实例。

```typescript
const { getInstance } = useLeaferAnnotateSingleton()
const instance = getInstance()
```

#### 元素操作

```typescript
const {
  delElement, // 删除元素
  selectMark, // 选择元素
  setMarkHover, // 设置悬停状态
  unsetMarkHover // 取消悬停状态
} = useLeaferAnnotateSingleton()

delElement('element-id')
selectMark('mark-id')
setMarkHover('mark-id')
unsetMarkHover('mark-id')
```

#### 事件监听

```typescript
const { onElementSelect, onElementAdd } = useLeaferAnnotateSingleton()

// 监听元素选择事件
const unsubscribeSelect = onElementSelect(event => {
  console.log('选中元素:', event.element)
  console.log('来源:', event.source)
})

// 监听元素添加事件
const unsubscribeAdd = onElementAdd(event => {
  console.log('添加元素:', event.element)
})

// 清理监听器
unsubscribeSelect()
unsubscribeAdd()
```

#### 响应式数据

```typescript
const { markList } = useLeaferAnnotateSingleton()

// markList 是响应式的标记列表
watch(markList, newList => {
  console.log('标记列表更新:', newList)
})
```

## useLeaferLimit

### 功能特性

- **本地存储**：使用 `useLocalStorage` 持久化配置
- **自动同步**：配置变更时自动同步到 Leafer 实例
- **响应式状态**：提供响应式的宽高和锁定状态

### API 接口

```typescript
const {
  width, // 宽度 (Ref<number>)
  height, // 高度 (Ref<number>)
  lockWidth, // 锁定宽度 (Ref<boolean>)
  lockHeight, // 锁定高度 (Ref<boolean>)
  setWidth, // 设置宽度
  setHeight, // 设置高度
  toggleLockWidth, // 切换宽度锁定
  toggleLockHeight // 切换高度锁定
} = useLeaferLimit()
```

### 响应式数据监听

```typescript
const { markList } = useLeaferAnnotateSingleton()

// 使用 watchEffect 自动追踪依赖
watchEffect(() => {
  console.log('当前标记数量:', markList.value.length)
})

// 或者使用 watch 进行精确控制
watch(
  markList,
  (newList, oldList) => {
    console.log('标记列表变化:', { newList, oldList })
  },
  { deep: true }
)
```

## 注意事项

1. **单例特性**：`useLeaferAnnotateSingleton` 返回的是单例，所有组件共享同一个实例
2. **自动销毁**：当最后一个使用该 Hook 的组件卸载时，实例会自动销毁
3. **事件总线**：使用 `@vueuse/core` 的事件总线，确保跨组件通信
4. **本地存储**：`useLeaferLimit` 的配置会自动保存到 localStorage
5. **类型安全**：所有 API 都有完整的 TypeScript 类型定义
