# 架构概览

## 1. 依赖关系
依赖链：[LeaferJS](https://www.leaferjs.com/) ← LeaferAnnotate ← Hook ← 页面

- **LeaferJS（底层引擎）**: 负责 Canvas 渲染、事件和交互
- **LeaferAnnotate（业务实例）**: 基于 LeaferJS 封装业务逻辑和方法
- **Hook（useLeaferAnnotateSingleton）**: 再次封装LeaferAnnotate,管理单例、暴露页面可用的方法
- **页面（Vue 组件）**: 仅通过 Hook 与标注实例交互

## 2. LeaferJS
```js
import { App, Rect } from 'leafer-ui'

const app = new App({
    view: window, 
    width: 600, 
    height: 600,
    fill: '#333'
})

app.tree.add(Rect.one({ fill: '#32cd79', draggable: true }, 100, 100))
```

## 3. LeaferAnnotate 类

```ts
export interface ILeaferAnnotate {
  config: LeaferAnnotateConfig
  app: App
  pageFrame: IFrame
  
  snap?: any
  limit: LimitConfig
  activeQuestionID: number | null
  
  /** 初始化实例 */
  init(): Promise<void>
  /** 重置视图到初始状态 */
  resetView(): void
  /**
   * 加载页面与标注数据
   */
  loadData(pageUrl: string, marks: IMark[]): Promise<void>

  delElement(id: string): void
  selectMark(id: string): void
  setMarkHover(id: string): void
  unsetMarkHover(id: string): void
  setActiveQuestionID(questionID: number | null): void
  getActiveQuestionID(): number | null
  changeMode(mode: 'view' | 'edit'): void
  destroy(): void
}
```


```ts
import { createLeaferAnnotate } from '../mark/leafer-mark-core/index'

// DOM 容器
const view = document.getElementById('annotate-view') as HTMLDivElement

// 创建实例工厂
const { getInstance, destroy } = await createLeaferAnnotate({
  view,
})

// 获取实例
const leaferAnnotate = getInstance()

// 加载底图与标注
await leaferAnnotate?.loadData('https://example.com/page.png', [
  {
    id: 'mark-1',
    x: 120,
    y: 80,
    width: 160,
    height: 60,
    questionID: 1001,
    color: '#FF6B6B'
  }
])

// 交互 API 示例
lealeaferAnnotatefer?.changeMode('view')
leaferAnnotate?.selectMark('mark-1')
leaferAnnotate?.setActiveQuestionID(1001)
leaferAnnotate?.resetView()

// 销毁
await destroy()
```

## 使用Hook包装 leaferAnnotate

下面示例基于页面中通过 Hook 演示如何在组件中初始化、加载底图与标注、选中标注以及重置视图。

```vue
<script setup lang="ts">
import { onMounted } from 'vue'
import { useLeaferAnnotateSingleton } from '../mark/leafer-mark-core/useLeaferAnnotate'

const { createApp, loadData, selectMark, resetView, markList } = useLeaferAnnotateSingleton()

onMounted(async () => {
  // 初始化实例，挂载到元素汇总
  await createApp({ view: 'leafer-container' })

  // 加载底图和框选
  await loadData('https://example.com/page.png', [
    {
      id: 'mark-1',
      x: 120,
      y: 80,
      width: 160,
      height: 60,
      questionID: 1001,
      color: '#FF6B6B'
    }
  ])
  selectMark('mark-1')
})
</script>
```