import { App, Image, Frame, Resource } from 'leafer-editor'
import { AdsorptionBinding } from './leafer.adsorption.js'
import { DropBinding } from './leafer.drop.js'
import { CopyRectBinding } from './leafer.copyRect.js'
import { CreateRectBinding } from './leafer.createRect.js'
import './leafer.proxyData.js'
import type { IFrame } from 'leafer-editor'
import '@leafer-in/state'
import { Ruler } from 'leafer-x-ruler'
import { Snap } from './plugins/snap/index.js'
import { loadImage, processMarksToRects, configureInteractionMode, getThemeColor } from './leafer.helper.js'
import {
  DEFAULT_LEAFER_CONFIG,
  DEFAULT_SNAP_CONFIG,
  DEFAULT_RULER_CONFIG,
  DEFAULT_RECT_CONFIG
} from './leafer.config.js'
import type { LeaferAnnotateConfig, IMark, ILeaferAnnotate, RectWithData, LimitConfig } from './leafer.type.js'
import { useLocalStorage } from '@vueuse/core'
import { cloneDeep } from 'lodash-es'
import './leafer.setStroke.js'
export class LeaferAnnotate implements ILeaferAnnotate {
  config: LeaferAnnotateConfig
  app!: App
  pageFrame!: IFrame
  snap!: Snap
  limit!: LimitConfig
  activeQuestionID: number | null = null
  private mode: 'view' | 'edit' = 'view'
  private adsorptionBinding!: AdsorptionBinding
  private dropBinding!: DropBinding
  private copyRectBinding!: CopyRectBinding
  private createRectBinding!: CreateRectBinding
  private ruler!: Ruler

  constructor(config: LeaferAnnotateConfig) {
    this.config = config
    const storage = useLocalStorage<LimitConfig>('leafer-limit', {
      width: 0,
      height: 0,
      lockWidth: false,
      lockHeight: false
    })
    this.limit = storage.value
  }

  async init(): Promise<void> {
    let { view } = this.config

    // 初始化画布，没有设置宽高，默认使用父元素的宽高
    this.app = new App({
      view: view,
      ...DEFAULT_LEAFER_CONFIG
    })

    // // 设置图片和标记
    this.pageFrame = new Frame({
      id: 'pageFrame'
    })
    this.app.tree?.add(this.pageFrame)

    // 启用插件
    this.bindPlugins()

    // 默认为view模式
    this.changeMode('view')
  }

  public destroy(): void {
    // 清理资源
    Resource.destroy()

    // 注销插件
    if (this.snap) {
      this.snap.destroy()
      this.snap = null as any
    }
    if (this.adsorptionBinding) {
      this.adsorptionBinding.uninstall()
      this.adsorptionBinding = null as any
    }
    if (this.dropBinding) {
      this.dropBinding.uninstall()
      this.dropBinding = null as any
    }
    if (this.copyRectBinding) {
      this.copyRectBinding.uninstall()
      this.copyRectBinding = null as any
    }
    if (this.createRectBinding) {
      this.createRectBinding.uninstall()
      this.createRectBinding = null as any
    }

    if (this.ruler) {
      this.ruler.enabled = false
      this.ruler = null as any
    }

    if (this.pageFrame) {
      this.pageFrame.removeAll()
      this.pageFrame = null as any
    }

    if (this.app) {
      this.app.destroy()
      this.app = null as any
    }

    this.config = null as any

    return
  }
  public selectMark(markId: string): void {
    let mark = this.pageFrame.findOne(`#${markId}`)
    if (mark) {
      this.app.editor.select(mark)
    } else {
      throw new Error(`Mark with id ${markId} not found`)
    }
  }

  /**
   * 设置标记悬停状态
   * @param markId 标记ID
   */
  public setMarkHover(markId: string): void {
    let mark: RectWithData = this.pageFrame.findOne(`#${markId}`) as RectWithData
    if (mark) {
      let { hoverStrokeColor } = getThemeColor(mark.data)
      mark.fill = hoverStrokeColor
    }
  }

  /**
   * 取消标记悬停状态
   * @param markId 标记ID
   */
  public unsetMarkHover(markId: string): void {
    let mark: RectWithData = this.pageFrame.findOne(`#${markId}`) as RectWithData
    if (mark) {
      let { activeStrokeColor } = getThemeColor(mark.data)
      mark.fill = activeStrokeColor
    }
  }

  /**
   * 设置当前激活的问题ID
   * @param questionID 问题ID
   */
  public setActiveQuestionID(questionID: number | null): void {
    this.activeQuestionID = questionID
  }

  /**
   * 获取当前激活的问题ID
   * @returns 当前激活的问题ID
   */
  public getActiveQuestionID(): number | null {
    return this.activeQuestionID
  }
  public resetView(): void {
    this.app.tree?.zoom('fit-width', [12 + 20, 12, 12, 12 + 20], false, true)
  }
  /**
   * 设置图片和标记
   */
  public async loadData(pageUrl: string, marks: IMark[]): Promise<void> {
    if (this.pageFrame) {
      this.pageFrame.clear()
      Resource.destroy()
    }
    // 获取底图，这是整个画布的核心
    let { url, width, height } = await loadImage(pageUrl, 'bg.png')

    this.pageFrame.width = width
    this.pageFrame.height = height

    this.app.tree?.zoom('fit-width', [12 + 20, 12, 12, 12 + 20])
    this.pageFrame.add(new Image({ id: 'bg-image', url: url, width: width, height: height }))

    // 初始化标注
    this.initMarks(marks)
  }

  /**
   * 根据ID删除元素
   * @param id 元素ID
   */
  delElement(markId: string): void {
    let mark: RectWithData = this.pageFrame.findOne(`#${markId}`) as RectWithData

    if (mark) {
      this.pageFrame.remove(mark)
    }
  }

  /**
   * 切换模式
   *
   * view: 可以缩放，移动页面，可以编辑元素
   *
   * edit: 不可以缩放，移动，不可以编辑元素
   * @param mode
   */
  public changeMode(mode: 'view' | 'edit'): void {
    this.mode = mode
    this.pageFrame.cursor = 'crosshair'
    let interaction = this.app.tree?.interaction
    configureInteractionMode(interaction, mode)

    if (this.mode === 'view') {
      this.pageFrame.hitChildren = true
    } else if (this.mode === 'edit') {
      this.pageFrame.hitChildren = false
    }
  }

  /**
   * 初始化接口中的标注矩形
   */
  private initMarks(marks: IMark[]): void {
    if (!Array.isArray(marks) || !this.pageFrame) return
    const layer = this.pageFrame
    const rects = processMarksToRects(marks)
    rects.forEach(rect => {
      layer.add(rect)
    })
  }

  private bindPlugins(): void {
    this.snap = new Snap(this.app, {
      ...DEFAULT_SNAP_CONFIG,
      parentContainer: this.pageFrame
    })
    this.ruler = new Ruler(this.app, DEFAULT_RULER_CONFIG)

    // 精确控制元素，x,y,width,height在1px单位步进变化,去除小数点
    this.adsorptionBinding = new AdsorptionBinding()
    this.adsorptionBinding.install(this.app)
    // 在画布上拖拽图元
    this.dropBinding = new DropBinding()
    this.dropBinding.install(this)
    // 按住ctrl复制矩形
    this.copyRectBinding = new CopyRectBinding()
    this.copyRectBinding.install(this, DEFAULT_RECT_CONFIG.className)
    // 创建矩形工具
    this.createRectBinding = new CreateRectBinding()
    this.createRectBinding.install(this)

    this.snap.enable(true)
    this.ruler.enabled = true
  }
}

export const createLeaferAnnotate = async (
  config: LeaferAnnotateConfig
): Promise<{
  getInstance: () => LeaferAnnotate | null
  destroy: () => Promise<void>
}> => {
  let instance: LeaferAnnotate | null = new LeaferAnnotate(cloneDeep(config))
  await instance.init()

  let isDestroyed = false

  return {
    getInstance: function () {
      if (isDestroyed) {
        console.warn('LeaferAnnotate instance has been destroyed')
        return null
      }
      return instance
    },
    destroy: async () => {
      if (instance && !isDestroyed) {
        await instance.destroy()
        instance = null
        isDestroyed = true
      }
    }
  }
}
