/*
 * @Author: zengzhe
 * @Date: 2025-08-18 18:53:16
 * @LastEditors: zengzhe
 * @LastEditTime: 2025-08-18 20:44:37
 * @Description:
 */
import { type IFrame, App, type Rect, type IRectInputData } from 'leafer-editor'

/**
 * 处理后的底图信息
 */
export interface IImageInfo {
  /** 图片的url, leafer Resource 的资源路径 */
  url: string
  /** 图片的宽度，单位px */
  width: number
  /** 图片的高度，单位px */
  height: number
}

// 接口中的坐标点类型
export interface IMarkPoint {
  x: number
  y: number
}

/**
 * 标注类型枚举
 */
export enum MarkType {
  /** 题目 */
  QUESTION = 1,
  /** 功能框 */
  FUNCTION_BOX = 2
}

/**
 * 关联资源信息
 */
export interface IMarkResource {
  /** 封面地址 */
  coverUrl: string
  /** 腾讯视频Id */
  fileId: string
  /** 视频名称 */
  fileName: string
  /** 文件大小kb */
  fileSize: number
  /** 1视频 2音频 */
  fileType: 1 | 2
  /** 腾讯播放地址 */
  mediaUrl: string
  /** 资源ID */
  resourceId: string
}
/**
 * 接口中使用的标注数据
 */
export interface IMark {
  /** 唯一标识 */
  markId: string
  /** 表示标引矩形框的左上角坐标(x,y)，坐标数值单位为px */
  topPx: IMarkPoint
  /** 表示标引矩形框的右下角坐标(x,y)，坐标数值单位为px */
  bottomPx: IMarkPoint
  /** 表示标引矩形框的左上角坐标(x,y)，坐标数值单位为mm */
  top: IMarkPoint
  /** 表示标引矩形框的右下角坐标(x,y)，坐标数值单位为mm */
  bottom: IMarkPoint
  /** 题目序号 */
  questionID: number
  /** 功能框编码 */
  childQuestionID: string
  /** 功能框名称 */
  funcName: string
  /** 功能框图标 */
  funcIcon: string
  /** 好题库-题目ID */
  yqId: string
  /** 好题库-题目ID */
  exerciseID: string
  /** 类型：1-题目；2-功能框 */
  type: MarkType
  /** 输入时间 */
  inputTime: string
  /** 关联腾讯云vod资源列表 */
  resources?: IMarkResource[]
}

/**
 * LeaferAnnotate 构造函数的配置项接口
 */
export interface LeaferAnnotateConfig {
  /** 挂载的视图 */
  view: string
  /** 背景图片的URL，需要支持跨域才能加载 */
  pageUrl?: string
  /** 标注数据数组 */
  marks?: IMark[]
  /** 元素添加回调函数 */
  onElementAdd?: (element: RectWithData) => void
  /** 元素选择回调函数 */
  onElementSelect?: (element: RectWithData) => void
}

export type ActiveTool = 'move' | 'rect' | 'drag' | ''

export interface IPoint {
  x: number
  y: number
}

// 定义 LeaferAnnotate 类的接口，使其结构更清晰
export interface ILeaferAnnotate {
  config: LeaferAnnotateConfig
  app: App
  pageFrame: IFrame
  snap?: any
  /** 宽高限制配置 */
  limit: LimitConfig
  /** 当前激活的问题ID */
  activeQuestionID: number | null
  resetView(): void
  loadData(pageUrl: string, marks: IMark[]): Promise<void>
  init(): Promise<void>
  delElement(id: string): void
  selectMark(id: string): void
  setMarkHover(id: string): void
  unsetMarkHover(id: string): void
  setActiveQuestionID(questionID: number | null): void
  getActiveQuestionID(): number | null
  changeMode(mode: 'view' | 'edit'): void
  destroy(): void
}

export type WithData = { data?: IMark }

/**
 * 带有业务数据的 Rect 类型
 */
export type RectWithData = Rect & WithData

/**
 * 在创建图形时，为输入结构的 data 提供类型提示
 */
export type RectInputWithData = Omit<IRectInputData, 'data'> & WithData

/**
 * createLeaferAnnotate 函数的返回值类型
 */
export interface ILeaferAnnotateManager {
  getInstance: () => ILeaferAnnotate | null
  destroy: () => void
}

/**
 * 限制配置
 */
export interface LimitConfig {
  width: number
  height: number
  lockWidth: boolean
  lockHeight: boolean
}
