# Leafer Mark Core 添加新功能指南

## 概述

根据功能类型的不同，有两种主要的扩展方式：

1. **Binding 方式**：用于监听 Leafer 事件和响应用户交互
2. **Class 扩展方式**：用于添加新的属性和方法到核心类

## 功能类型判断

### 使用 Binding 的场景

当您需要以下功能时，应该使用 Binding 方式：

- 监听 Leafer 编辑器的事件（如 `pointerdown`、`pointermove`、`pointerup` 等）
- 响应用户的交互操作（如拖拽、点击、键盘事件等）
- 处理元素的创建、修改、删除等生命周期事件
- 实现与画布交互相关的功能

### 使用 Class 扩展的场景

当您需要以下功能时，应该直接扩展核心类：

- 添加新的配置属性
- 添加新的公共方法供外部调用
- 添加新的数据存储属性
- 添加新的工具方法
- 添加新的状态管理

## 方式一：使用 Binding 添加新功能

### 1. 创建 Binding 文件

创建 `leafer.yourFeature.ts` 文件：

```typescript
import { App, type IFrame } from 'leafer-editor'
import type { ILeaferAnnotate } from './leafer.type'

/**
 * 你的功能描述
 * 例如：处理用户双击创建文本标注
 */
export class YourFeatureBinding {
  private _instance!: ILeaferAnnotate
  private _boundEventHandler!: (e: any) => void

  /**
   * 安装服务
   */
  public install(instance: ILeaferAnnotate) {
    this._instance = instance
    this._boundEventHandler = this.onEventHandler.bind(this)

    // 绑定事件监听
    this._instance.app.on('YourEvent', this._boundEventHandler)
  }

  /**
   * 卸载服务
   */
  public uninstall() {
    if (this._instance.app && this._boundEventHandler) {
      this._instance.app.off('YourEvent', this._boundEventHandler)
    }
    this._boundEventHandler = null as any
    this._instance = null as any
  }

  /**
   * 事件处理函数
   */
  private onEventHandler(e: any) {
    // 你的业务逻辑
    console.log('Your feature triggered', e)
  }
}
```

### 2. 在主文件中注册

在 `index.ts` 中添加：

```typescript
// 导入你的 Binding
import { YourFeatureBinding } from './leafer.yourFeature.js'

export class LeaferAnnotate implements ILeaferAnnotate {
  // 添加私有属性
  private yourFeatureBinding!: YourFeatureBinding

  private bindPlugins(): void {
    // 其他插件...

    // 初始化你的功能
    this.yourFeatureBinding = new YourFeatureBinding()
    this.yourFeatureBinding.install(this)
  }

  public async destroy(): Promise<void> {
    // 其他清理...

    // 清理你的功能
    if (this.yourFeatureBinding) {
      this.yourFeatureBinding.uninstall()
      this.yourFeatureBinding = null as any
    }
  }
}
```

### 3. 可选：添加配置

在 `leafer.config.ts` 中添加配置：

```typescript
export const DEFAULT_YOUR_FEATURE_CONFIG = {
  enabled: true
  // 其他配置项
}
```

### 4. 可选：添加类型定义

在 `leafer.type.ts` 中添加类型：

```typescript
export interface YourFeatureConfig {
  enabled: boolean
  // 其他配置项
}
```

## 方式二：扩展核心类添加新功能

### 1. 添加新的属性

在 `index.ts` 的 `LeaferAnnotate` 类中添加：

```typescript
export class LeaferAnnotate implements ILeaferAnnotate {
  // 现有属性...

  // 新增属性
  private yourNewProperty: string = ''
  private yourNewData: any[] = []

  // 现有方法...
}
```

### 2. 添加新的公共方法

```typescript
export class LeaferAnnotate implements ILeaferAnnotate {
  // 现有方法...

  /**
   * 你的新方法描述
   * @param param1 参数1描述
   * @param param2 参数2描述
   * @returns 返回值描述
   */
  public yourNewMethod(param1: string, param2?: number): boolean {
    // 你的业务逻辑
    return true
  }

  /**
   * 获取你的新数据
   */
  public getYourNewData(): any[] {
    return this.yourNewData
  }

  /**
   * 设置你的新属性
   */
  public setYourNewProperty(value: string): void {
    this.yourNewProperty = value
  }
}
```

### 3. 更新接口定义

在 `leafer.type.ts` 中更新 `ILeaferAnnotate` 接口：

```typescript
export interface ILeaferAnnotate {
  // 现有方法...

  // 新增方法
  yourNewMethod(param1: string, param2?: number): boolean
  getYourNewData(): any[]
  setYourNewProperty(value: string): void
}
```

### 4. 在构造函数中初始化

```typescript
export class LeaferAnnotate implements ILeaferAnnotate {
  constructor(config: LeaferAnnotateConfig) {
    // 现有初始化...

    // 初始化新属性
    this.yourNewProperty = config.yourNewProperty || ''
    this.yourNewData = []
  }
}
```

### 5. 更新配置接口

在 `leafer.type.ts` 中更新配置接口：

```typescript
export interface LeaferAnnotateConfig {
  // 现有配置...

  // 新增配置
  yourNewProperty?: string
  yourNewFeature?: YourFeatureConfig
}
```

## 使用示例

### Binding 方式示例

```typescript
// 创建实例时，你的功能会自动启用
const annotate = await createLeaferAnnotate({
  view: '#canvas',
  pageUrl: 'image.jpg',
  marks: []
})

// 功能会自动工作，无需额外配置
```

### Class 扩展方式示例

```typescript
// 创建实例
const annotate = await createLeaferAnnotate({
  view: '#canvas',
  pageUrl: 'image.jpg',
  marks: [],
  yourNewProperty: '初始值'
})

// 调用新方法
const result = annotate.yourNewMethod('参数1', 123)
const data = annotate.getYourNewData()

// 设置新属性
annotate.setYourNewProperty('新值')
```

## 混合使用示例

有时候一个功能既需要监听事件，又需要提供新的方法，可以同时使用两种方式：

```typescript
// 1. 在核心类中添加新方法
export class LeaferAnnotate implements ILeaferAnnotate {
  private yourFeatureBinding!: YourFeatureBinding
  private yourFeatureData: any[] = []

  /**
   * 获取功能数据
   */
  public getYourFeatureData(): any[] {
    return this.yourFeatureData
  }

  /**
   * 手动触发功能
   */
  public triggerYourFeature(): void {
    // 手动触发逻辑
  }
}

// 2. 在 Binding 中监听事件并更新数据
export class YourFeatureBinding {
  private _instance!: ILeaferAnnotate

  public install(instance: ILeaferAnnotate) {
    this._instance = instance
    this._instance.app.on('pointerdown', this.onPointerDown.bind(this))
  }

  private onPointerDown(e: any) {
    // 处理事件并更新核心类的数据
    // 通过实例访问核心类的方法和属性
  }
}
```

## 注意事项

### Binding 方式注意事项

1. **命名规范**: 文件名使用 `leafer.featureName.ts` 格式
2. **类名规范**: 使用 `FeatureNameBinding` 格式
3. **事件清理**: 必须在 `uninstall()` 中清理所有事件监听
4. **错误处理**: 添加适当的错误处理和边界检查
5. **文档注释**: 为公共方法添加 JSDoc 注释

### Class 扩展注意事项

1. **接口同步**: 新增公共方法时，必须同步更新 `ILeaferAnnotate` 接口
2. **配置同步**: 新增配置项时，必须同步更新 `LeaferAnnotateConfig` 接口
3. **初始化**: 新属性需要在构造函数中正确初始化
4. **类型安全**: 确保所有新增的方法和属性都有正确的类型定义
5. **向后兼容**: 新增的配置项应该提供默认值，确保向后兼容

### 通用注意事项

1. **功能单一**: 每个 Binding 或方法应该只负责一个明确的功能
2. **错误处理**: 添加适当的错误处理和边界检查
3. **性能考虑**: 避免在事件处理函数中执行耗时操作
4. **内存管理**: 确保在 `destroy()` 方法中正确清理资源
5. **测试**: 为新功能编写相应的测试用例

## 常见问题

### Q: 什么时候使用 Binding，什么时候扩展 Class？

A:

- 需要监听 Leafer 事件或响应用户交互时使用 Binding
- 需要提供新的 API 方法或存储新数据时扩展 Class
- 复杂功能可以同时使用两种方式

### Q: 如何确保新功能不会影响现有功能？

A:

- 遵循现有的命名规范
- 添加适当的错误处理
- 在 `destroy()` 方法中正确清理资源
- 为新功能编写测试用例

### Q: 如何调试新功能？

A:

- 使用 `console.log` 输出调试信息
- 利用浏览器开发者工具的事件监听器
- 检查 Leafer 编辑器的事件文档
