import { PointerEvent, Rect, EditorEvent, IUI } from 'leafer-editor'
import type { ILeaferAnnotate, RectWithData } from './leafer.type'
import { calculateRectBounds, createRect, createMarkData, isValidRectSize, roundPoint } from './leafer.helper'
import { DEFAULT_RECT_CONFIG } from './leafer.config'
import { nanoid } from 'nanoid'
/**
 * 创建矩形
 */
export class CreateRectBinding {
  private _instance!: ILeaferAnnotate
  private isCreating = false
  private startPoint: { x: number; y: number } | null = null
  private isElementSelected = false
  private previewRect: Rect | null = null

  private editorSelect: ((e: EditorEvent) => void) | null = null
  private pageFrameDown: ((e: PointerEvent) => void) | null = null
  private pageFrameMove: ((e: PointerEvent) => void) | null = null
  private pageFrameUp: ((e: PointerEvent) => void) | null = null

  /**
   * 加载服务
   */
  public install(instance: ILeaferAnnotate) {
    this._instance = instance
    this.bindEvents()
  }

  /**
   * 卸载服务
   */
  public uninstall() {
    this.unbindEvents()

    if (this.previewRect) {
      this.previewRect.remove()
      this.previewRect = null
    }

    this.startPoint = null
    this.isCreating = false
    this.isElementSelected = false
    this._instance = null as unknown as ILeaferAnnotate
  }

  private bindEvents() {
    this.editorSelect = (e: EditorEvent) => {
      let target = e.value as IUI
      let list = this._instance.pageFrame.find(('.' + DEFAULT_RECT_CONFIG.className) as string)
      list.forEach(item => {
        item.set({ hitSelf: false, zIndex: 1 })
      })
      if (target) {
        target.set({ hitSelf: true, zIndex: 10 })

        this.isElementSelected = true
        if (target?.className === DEFAULT_RECT_CONFIG.className && this._instance.config?.onElementSelect) {
          this._instance.config.onElementSelect(target as RectWithData)
        }
      } else {
        this.isElementSelected = false
      }
    }

    this.pageFrameDown = e => {
      if (this.isElementSelected) return
      this.isCreating = true
      this.startPoint = roundPoint(this._instance.pageFrame.getLocalPoint(e))
      this._instance.changeMode('edit')
    }

    this.pageFrameMove = e => {
      if (this.isElementSelected) return
      if (this.isCreating && this.startPoint) {
        const currentPoint = roundPoint(this._instance.pageFrame.getLocalPoint(e))
        const { x, y, width, height } = calculateRectBounds(this.startPoint, currentPoint)

        const limitedWidth = this._instance.limit.lockWidth ? Math.min(width, this._instance.limit.width) : width
        const limitedHeight = this._instance.limit.lockHeight ? Math.min(height, this._instance.limit.height) : height

        if (isValidRectSize(limitedWidth, limitedHeight)) {
          if (!this.previewRect) {
            this.previewRect = createRect({
              id: 'preview-rect',
              x,
              y,
              width: limitedWidth,
              height: limitedHeight
            })
            this._instance.pageFrame.add(this.previewRect)
            if (this._instance.snap) {
              this._instance.snap.triggerSnap(this.previewRect)
            }
          } else {
            this.previewRect.set({ x, y, width: limitedWidth, height: limitedHeight })
            if (this._instance.snap) {
              this._instance.snap.triggerSnap(this.previewRect)
            }
          }
        } else if (this.previewRect) {
          this.previewRect.remove()
          this.previewRect = null
          if (this._instance.snap) {
            this._instance.snap.destroy()
          }
        }
      }
    }

    this.pageFrameUp = e => {
      if (this.isElementSelected) return
      if (this.isCreating && this.startPoint) {
        const currentPoint = roundPoint(this._instance.pageFrame.getLocalPoint(e))
        const { x, y, width, height } = calculateRectBounds(this.startPoint, currentPoint)

        if (this.previewRect) {
          this.previewRect.remove()
          this.previewRect = null
        }

        if (this._instance.snap) {
          this._instance.snap.destroy()
        }

        const limitedWidth = this._instance.limit.lockWidth ? Math.min(width, this._instance.limit.width) : width
        const limitedHeight = this._instance.limit.lockHeight ? Math.min(height, this._instance.limit.height) : height

        if (isValidRectSize(limitedWidth, limitedHeight)) {
          let markId = nanoid()
          const rect = createRect({
            id: markId,
            x,
            y,
            width: limitedWidth,
            height: limitedHeight,
            data: createMarkData({
              questionID: this._instance.getActiveQuestionID() || 0,
              markId
            })
          })
          this._instance.pageFrame.add(rect)
          this._instance.selectMark(markId)
          // updateMarkStyleByData(rect, rect.data)
          if (this._instance.config?.onElementAdd) {
            this._instance.config.onElementAdd(rect)
          }
        }

        this.isCreating = false
        this._instance.changeMode('view')
        this.startPoint = null
      }
    }

    this._instance.app.editor.on(EditorEvent.SELECT, this.editorSelect)
    this._instance.pageFrame.on(PointerEvent.DOWN, this.pageFrameDown)
    this._instance.pageFrame.on(PointerEvent.MOVE, this.pageFrameMove)
    this._instance.pageFrame.on(PointerEvent.UP, this.pageFrameUp)
  }

  private unbindEvents() {
    if (this._instance.app?.editor && this.editorSelect) {
      this._instance.app.editor.off(EditorEvent.SELECT, this.editorSelect)
    }
    if (this._instance.pageFrame && this.pageFrameDown) {
      this._instance.pageFrame.off(PointerEvent.DOWN, this.pageFrameDown)
    }
    if (this._instance.pageFrame && this.pageFrameMove) {
      this._instance.pageFrame.off(PointerEvent.MOVE, this.pageFrameMove)
    }
    if (this._instance.pageFrame && this.pageFrameUp) {
      this._instance.pageFrame.off(PointerEvent.UP, this.pageFrameUp)
    }

    this.editorSelect = null
    this.pageFrameDown = null
    this.pageFrameMove = null
    this.pageFrameUp = null
  }
}
