/*
 * @Author: zengzhe
 * @Date: 2025-08-18 18:53:16
 * @LastEditors: zengzhe
 * @LastEditTime: 2025-08-18 21:49:59
 * @Description:
 */
import { Resource, Rect, type IInteraction } from 'leafer-editor'
import { MarkType } from './leafer.type'
import type { IMark, IImageInfo, IPoint, RectInputWithData, RectWithData, IMarkPoint } from './leafer.type'
import { toRaw } from 'vue'
import { DEFAULT_RECT_CONFIG, CANVAS_SELECTOR, MIN_RECT_SIZE, DEFAULT_THEME } from './leafer.config'
import dayjs from 'dayjs'
import { nanoid } from 'nanoid'

//#region 工具函数

/**
 * 格式化时间为 YYYY-MM-DD HH:mm:ss 格式
 */
export function formatDateTime(date: Date = new Date()): string {
  return dayjs(date).format('YYYY-MM-DD HH:mm:ss')
}

/**
 * 四舍五入坐标点到整数
 * @param point 坐标点
 *
 * @example
 * const point = { x: 1.2, y: 2.5 };
 * const roundedPoint = roundPoint(point);
 * console.log(roundedPoint); // { x: 1, y: 3 }
 * @returns 四舍五入后的坐标点
 */
export function roundPoint(point: IPoint): IPoint {
  return {
    x: Math.round(point.x),
    y: Math.round(point.y)
  }
}

/**
 * 设置一个对象的属性值
 * @param obj 对象实例
 * @param key 属性
 * @param val 值
 */
export function assign<T extends object, K extends keyof T>(obj: T, key: K, val: T[K]) {
  obj[key] = val // okay
}

/**
 * 计算矩形边界
 */
export function calculateRectBounds(startPoint: { x: number; y: number }, currentPoint: { x: number; y: number }) {
  const x = Math.min(startPoint.x, currentPoint.x)
  const y = Math.min(startPoint.y, currentPoint.y)
  const width = Math.abs(currentPoint.x - startPoint.x)
  const height = Math.abs(currentPoint.y - startPoint.y)

  return { x, y, width, height }
}

/**
 * 创建元素数据
 */
export function createMarkData(data?: Partial<IMark>): IMark {
  const defaultData: IMark = {
    markId: nanoid(),
    topPx: { x: 0, y: 0 },
    bottomPx: { x: 0, y: 0 },
    top: { x: 0, y: 0 },
    bottom: { x: 0, y: 0 },
    questionID: 0,
    childQuestionID: '',
    funcName: '',
    funcIcon: '',
    yqId: '',
    exerciseID: '',
    type: MarkType.QUESTION,
    inputTime: formatDateTime(),
    resources: []
  }

  return {
    ...defaultData,
    ...data
  }
}
/**
 * 更新样式并重新选中元素（确保样式立即生效）
 */
export function updateMarkStyleAndReselect(target: RectWithData) {
  const editor = target.app?.editor
  // 更新样式
  updateMarkStyleByData(target)

  if (editor.hasItem(target)) {
    // 先取消选中
    editor?.cancel()
    // 重新选中以应用新样式
    editor?.select(target)
  }
}
export function getThemeColor(data: IMark) {
  let { exerciseID } = data
  let id = exerciseID
  const strokeColor = id ? DEFAULT_THEME.stroke : DEFAULT_THEME.unSetDataStroke
  const hoverStrokeColor = id ? DEFAULT_THEME.hoverStroke : DEFAULT_THEME.unSetDataHoverStroke
  const activeStrokeColor = id ? DEFAULT_THEME.activeStroke : DEFAULT_THEME.unSetDataActiveStroke
  return { strokeColor, hoverStrokeColor, activeStrokeColor }
}
/**
 * 更新元素样式
 */
export function updateMarkStyleByData(target: RectWithData) {
  const { strokeColor, activeStrokeColor } = getThemeColor(target.data)
  // 1. 更新基础样式
  target.stroke = strokeColor
  target.hoverStyle = {
    fill: activeStrokeColor,
    stroke: strokeColor
  }
  target.fill = activeStrokeColor
  // 2. 更新编辑器配置
  target.editConfig = {
    hoverStyle: {
      fill: activeStrokeColor,
      stroke: strokeColor
    },
    selectedStyle: {
      fill: activeStrokeColor,
      stroke: strokeColor
    },
    stroke: strokeColor
  }
  target.forceUpdate()
}

/**
 * 检查矩形尺寸是否有效
 */
export function isValidRectSize(width: number, height: number, minSize: number = MIN_RECT_SIZE) {
  return width > minSize && height > minSize
}

/**
 * 获取Canvas元素
 */
export function getCanvasElement(selector: string = CANVAS_SELECTOR): HTMLElement {
  return document.querySelector(selector) as HTMLElement
}

/**
 * 解析拖拽数据
 */
export function parseDragData(dataTransfer: DataTransfer | null) {
  try {
    const dataString = dataTransfer?.getData('application/json')
    if (!dataString) return null

    const shapeData = JSON.parse(dataString)
    if (shapeData.source !== 'toolbar') return null

    return shapeData
  } catch (error) {
    return null
  }
}

/**
 * 配置交互模式
 */
export function configureInteractionMode(interaction: IInteraction | undefined, mode: 'view' | 'edit') {
  if (!interaction) return

  if (mode === 'view') {
    interaction.config.move!.holdMiddleKey = true
    interaction.config.move!.holdSpaceKey = true
    interaction.config.wheel!.disabled = false
  } else {
    interaction.config.move!.drag = false
    interaction.config.move!.holdMiddleKey = false
    interaction.config.move!.holdSpaceKey = false
    interaction.config.wheel!.disabled = true
  }
}
//#endregion

export function loadImage(src: string, name: string): Promise<IImageInfo> {
  return new Promise((resolve, reject) => {
    const image = new window.Image()
    if (src.includes(';base64')) {
      image.src = src
    } else {
      if (!src.includes('?')) {
        src += '?time=' + new Date().valueOf()
      } else {
        src += '&time=' + new Date().valueOf()
      }
      image.src = src
      image.crossOrigin = 'Anonymous'
    }

    image.onload = () => {
      const { url } = Resource.setImage(`leafer://${name}`, image)
      resolve({
        url: url,
        width: image.width,
        height: image.height
      })
    }
    image.onerror = () => {
      console.error('底图加载失败')
      reject('file load iamge')
    }
  })
}

//#region 初始化标注

/**
 * 将接口给的标注信息处理成leafer rect
 * @param marks 标注信息
 * @returns leafer rects
 */
export function processMarksToRects(marks: IMark[]): Rect[] {
  return marks
    .filter(mark => {
      const hasPoint = mark && mark.topPx && mark.bottomPx
      if (!hasPoint) return false
      return true
    })
    .map(mark => {
      const left = Math.min(mark.topPx.x, mark.bottomPx.x)
      const right = Math.max(mark.topPx.x, mark.bottomPx.x)
      const top = Math.min(mark.topPx.y, mark.bottomPx.y)
      const bottom = Math.max(mark.topPx.y, mark.bottomPx.y)

      const p1 = { x: left, y: top }
      const p2 = { x: right, y: bottom }
      return createRect({
        id: mark.markId,
        x: p1.x,
        y: p1.y,
        width: Math.max(0, Math.abs(p2.x - p1.x)),
        height: Math.max(0, Math.abs(p2.y - p1.y)),
        data: toRaw<IMark>({
          ...mark
        })
      })
    })
}

/**
 * 将 x, y, width, height 转换为 top 和 bottom 坐标点
 * @param x 矩形左上角 x 坐标
 * @param y 矩形左上角 y 坐标
 * @param width 矩形宽度
 * @param height 矩形高度
 * @returns 包含 top 和 bottom 坐标点的对象
 */
export function processRectToMarkPoints(
  x: number,
  y: number,
  width: number,
  height: number
): {
  top: IMarkPoint
  bottom: IMarkPoint
} {
  return {
    top: { x, y },
    bottom: { x: x + width, y: y + height }
  }
}

/**
 * 将 px 坐标点转换为 mm 坐标点
 * @param point 坐标点
 * @param ppi ppi
 * @param mm 毫米
 * @returns
 */
export function processPxToMmPoint(point: IMarkPoint, ppi: number = 300, mm: number = 24.5): IMarkPoint {
  return {
    x: (point.x / ppi) * mm,
    y: (point.y / ppi) * mm
  }
}

//#endregion

//#region 默认rect图形配置
/**
 * 创建一个rect
 *
 * 需要填充x,y,width,height，data业务属性
 */
/**
 * 创建一个 Rect，并为 data 提供类型提示
 * @param data 位置尺寸与业务数据
 */
export const createRect = (data: RectInputWithData): RectWithData => {
  return new Rect({
    ...DEFAULT_RECT_CONFIG,
    ...data
  }) as RectWithData
}

//#endregion
