import { ref, onUnmounted, computed, watch } from 'vue'
import type { Ref, ComputedRef } from 'vue'
import { useEventBus, type EventBusKey } from '@vueuse/core'
import { useLocalStorage } from '@vueuse/core'
import { createLeaferAnnotate as createFactory } from './index'
import type {
  LeaferAnnotateConfig,
  ILeaferAnnotate,
  ILeaferAnnotateManager,
  RectWithData,
  IMark,
  LimitConfig
} from './core/leafer.type'

interface ElementEvent {
  element: RectWithData
  source: string
}

const ELEMENT_SELECT_KEY: EventBusKey<ElementEvent> = Symbol('leafer-element-select')
const ELEMENT_ADD_KEY: EventBusKey<ElementEvent> = Symbol('leafer-element-add')

const __singletonManager = ref<ILeaferAnnotateManager | null>(null)
/**
 * 手动触发标记列表
 *
 * 因为instance无法改造成响应式，所以需要手动触发标记列表变化
 */
const listChangeTrigger = ref(0)

/**
 * 单例消费者令牌，按组件实例维度计数，用于控制最后一个使用者销毁时机
 */
let __consumerTokens = new Set<symbol>()

/**
 * 单例 Hook：跨多个子组件共享同一 Leafer 标注实例
 * - 支持引用计数，最后一个使用者卸载时自动销毁
 * - 支持基于事件总线的多组件事件通信
 */
export function useLeaferAnnotateSingleton(): {
  /**
   * 创建标注实例
   * @param config
   */
  createApp: (config: LeaferAnnotateConfig) => Promise<ILeaferAnnotateManager>
  /**
   * 加载底图和mark标注
   * @param pageUrl
   * @param marks
   */
  loadData: (pageUrl: string, marks: IMark[]) => Promise<void>
  /**
   * 销毁实例
   */
  destroy: () => void
  /**
   * 获取实例
   */
  getInstance: () => ILeaferAnnotate | null
  /**
   * 重置视图
   */
  resetView: () => void
  /**
   * 删除元素
   */
  delElement: (id: string) => void
  /**
   * 选择元素
   */
  selectMark: (id: string) => void
  /**
   * 设置标记悬停状态
   */
  setMarkHover: (id: string) => void
  /**
   * 取消标记悬停状态
   */
  unsetMarkHover: (id: string) => void
  /**
   * 设置当前激活的问题ID
   */
  setActiveQuestionID: (questionID: number | null) => void
  /**
   * 获取当前激活的问题ID
   */
  getActiveQuestionID: () => number | null
  /**
   * 监听元素选择事件
   */
  onElementSelect: (listener: (event: ElementEvent) => void) => () => void
  /**
   * 监听元素添加事件
   */
  onElementAdd: (listener: (event: ElementEvent) => void) => () => void
  /**
   * 标记列表
   */
  markList: ComputedRef<RectWithData[]>
  /**
   * 清空所有标记
   */
  clearPageData: () => void
} {
  /**
   * 单例消费者令牌，按组件实例维度计数，用于控制最后一个使用者销毁时机
   */
  const token = Symbol('leafer-consumer')
  __consumerTokens.add(token)

  const elementSelectBus = useEventBus(ELEMENT_SELECT_KEY)
  const elementAddBus = useEventBus(ELEMENT_ADD_KEY)

  /**
   * 触发元素选择事件
   */
  const emitElementSelect = (element: RectWithData, source: string = 'unknown') => {
    elementSelectBus.emit({ element, source })
  }

  /**
   * 触发元素添加事件
   */
  const emitElementAdd = (element: RectWithData, source: string = 'unknown') => {
    elementAddBus.emit({ element, source })
  }

  /**
   * 创建标注实例并自动配置事件总线
   */
  const createApp = async (config: LeaferAnnotateConfig) => {
    if (!__singletonManager.value) {
      const enhancedConfig = {
        ...config,
        onElementSelect: (element: RectWithData) => {
          emitElementSelect(element, 'leafer-core')
        },
        onElementAdd: (element: RectWithData) => {
          emitElementAdd(element, 'leafer-core')
          listChangeTrigger.value++
        }
      }
      __singletonManager.value = await createFactory(enhancedConfig)
    }
    return __singletonManager.value
  }

  /**
   * 加载底图及标注数据
   * @param pageUrl 页面图片地址
   * @param marks   标注数据集合
   */
  const loadData = async (pageUrl: string, marks: IMark[]) => {
    const instance = getInstance()
    if (instance) await instance.loadData(pageUrl, marks)
    listChangeTrigger.value++
  }

  /**
   * 销毁实例并清理事件监听器
   */
  const destroy = () => {
    __consumerTokens.delete(token)
    if (__consumerTokens.size === 0 && __singletonManager.value) {
      console.log('leafer destroyed')
      void __singletonManager.value.destroy()
      __singletonManager.value = null
      resetEvents()
    }
  }

  /**
   * 获取底层 Leafer 实例
   * @returns ILeaferAnnotate | null
   */
  const getInstance = (): ILeaferAnnotate | null => {
    return __singletonManager.value?.getInstance() ?? null
  }

  /**
   * 标记列表的响应式计算结果
   */
  const markList = computed<RectWithData[]>(() => {
    const instance = getInstance()
    if (!instance?.pageFrame) return []
    if (listChangeTrigger.value < 0) return []

    try {
      return instance.pageFrame.find('.mark').map(v => v.proxyData as RectWithData)
    } catch (error) {
      console.error('获取标记列表时出错:', error)
      return []
    }
  })

  /**
   * 清空所有标记
   */
  const clearPageData = (): void => {
    const inst = getInstance()
    if (!inst?.pageFrame) return
    try {
      inst.pageFrame.clear()
      listChangeTrigger.value++
    } catch (error) {
      console.error('清空标记列表时出错:', error)
    }
  }

  /**
   * 重置视图至初始状态
   */
  const resetView = (): void => {
    const inst = getInstance()
    if (inst) inst.resetView()
  }

  /**
   * 删除元素并触发删除事件
   */
  const delElement = (id: string): void => {
    const inst = getInstance()
    if (inst) {
      inst.delElement(id)
      listChangeTrigger.value++
    }
  }

  /**
   * 监听元素选择事件
   */
  const onElementSelect = (listener: (event: ElementEvent) => void) => {
    return elementSelectBus.on(listener)
  }

  /**
   * 监听元素添加事件
   */
  const onElementAdd = (listener: (event: ElementEvent) => void) => {
    return elementAddBus.on(listener)
  }
  /**
   * 选择元素
   */
  const selectMark = (id: string) => {
    const inst = getInstance()
    if (inst) inst.selectMark(id)
  }

  /**
   * 设置标记悬停状态
   */
  const setMarkHover = (id: string) => {
    const inst = getInstance()
    if (inst) inst.setMarkHover(id)
  }

  /**
   * 取消标记悬停状态
   */
  const unsetMarkHover = (id: string) => {
    const inst = getInstance()
    if (inst) inst.unsetMarkHover(id)
  }

  /**
   * 设置当前激活的问题ID
   */
  const setActiveQuestionID = (questionID: number | null) => {
    const inst = getInstance()
    if (inst) inst.setActiveQuestionID(questionID)
  }

  /**
   * 获取当前激活的问题ID
   */
  const getActiveQuestionID = (): number | null => {
    const inst = getInstance()
    return inst ? inst.getActiveQuestionID() : null
  }

  /**
   * 重置所有事件监听器
   */
  const resetEvents = () => {
    elementSelectBus.reset()
    elementAddBus.reset()
  }

  onUnmounted(() => {
    void destroy()
  })

  return {
    createApp,
    loadData,
    destroy,
    getInstance,
    resetView,
    delElement,
    onElementSelect,
    onElementAdd,
    markList,
    clearPageData,
    selectMark,
    setMarkHover,
    unsetMarkHover,
    setActiveQuestionID,
    getActiveQuestionID
  }
}

/**
 * 管理 Leafer 的宽高限制与锁定状态
 */
export function useLeaferLimit(): {
  width: Ref<number>
  height: Ref<number>
  lockWidth: Ref<boolean>
  lockHeight: Ref<boolean>
  setWidth: (v: number) => void
  setHeight: (v: number) => void
  toggleLockWidth: () => void
  toggleLockHeight: () => void
} {
  const store = useLocalStorage<LimitConfig>('leafer-limit', {
    width: 0,
    height: 0,
    lockWidth: false,
    lockHeight: false
  })

  const width = ref<number>(store.value.width)
  const height = ref<number>(store.value.height)
  const lockWidth = ref<boolean>(store.value.lockWidth)
  const lockHeight = ref<boolean>(store.value.lockHeight)

  const { getInstance } = useLeaferAnnotateSingleton()

  const syncToInstance = () => {
    const inst = getInstance()
    if (inst) {
      inst.limit = {
        width: width.value,
        height: height.value,
        lockWidth: lockWidth.value,
        lockHeight: lockHeight.value
      }
    }
  }

  watch([width, height, lockWidth, lockHeight], () => {
    store.value = {
      width: width.value,
      height: height.value,
      lockWidth: lockWidth.value,
      lockHeight: lockHeight.value
    }
    syncToInstance()
  })

  const setWidth = (v: number) => (width.value = Math.max(0, v || 0))
  const setHeight = (v: number) => (height.value = Math.max(0, v || 0))
  const toggleLockWidth = () => (lockWidth.value = !lockWidth.value)
  const toggleLockHeight = () => (lockHeight.value = !lockHeight.value)

  syncToInstance()

  return { width, height, lockWidth, lockHeight, setWidth, setHeight, toggleLockWidth, toggleLockHeight }
}
