
import { cloneDeep, get } from 'lodash'
import Konva from 'konva'
const toolImage4 = require('./tools-image/tool-icon-4.svg')
const toolImage5 = require('./tools-image/tool-icon-5.svg')
const toolImage6 = require('./tools-image/tool-icon-6.svg')
const toolImage7 = require('./tools-image/tool-icon-7.svg')
let imageList = {
  toolImage4,
  toolImage5,
  toolImage6,
  toolImage7
}
export class KonvaCanvas {
  /**
   * @constructor
   * @param {KonvaConfig} config
   */
  constructor(config) {
    let { stageBg, debug, defaultStore, defaultRotation } = config
    this.config = cloneDeep(config)
    this.debug = debug // 调试模式
    this.cursor = ''
    this.defaultRotation = defaultRotation
    this.stageBg = stageBg
    this.defaultStore = defaultStore
    this.points = config.points || []
    this.marks = config.marks || []
    this.containerWidth = config.containerWidth
    this.containerHeight = config.containerHeight
    //----------工具栏-------------S-//
    this.panelConfig = {
      strokeWidth: 2,
      stroke: '#FA5151'
    }
    this.startPointer = { x: 0, y: 0 }
    this.activeTool = null
    //----------工具栏-------------S-//

    //----------历史记录-------------S-//
    this.maxHistoryLength = 20
    this.history = [[], []]
    this.historyIndex = -1
    //----------历史记录-------------E-//
  }
  async init() {
    //1.回显状态
    this.store = []
    if (this.defaultStore && this.defaultStore.length) {
      this.store = this.defaultStore
      // 异步加载所有图片，全部加载完成后再继续
      const loadImagePromises = this.store.map(item => {
        if (item.imageUrl) {
          return loadImage(imageList[item.imageUrl]).then(({ image }) => {
            let index = this.store.findIndex(v => v.id == item.id)
            this.config.getCompThis().$set(this.store, index, {
              ...this.store[index],
              image: image
            })
          })
        } else {
          return Promise.resolve()
        }
      })
      await Promise.all(loadImagePromises)
      this.history = [[], cloneDeep(this.store)]
    }
    // 设置舞台大小 和 背景图标宽高
    const maxEdge = 840
    const { stageBg } = this.config

    // 加载背景图像
    const { image, width, height } = await loadImage(stageBg)

    // 计算缩放比例，使背景图最大边为 maxEdge
    const scale = Math.min(maxEdge / width, maxEdge / height, 1)
    const newWidth = width * scale
    const newHeight = height * scale

    // 舞台宽高 = 背景图宽高
    const stageWidth = newWidth
    const stageHeight = newHeight
    console.log(maxEdge)
    console.log(newWidth, newHeight)
    // 背景图层配置，居中
    this.bgLayerConfig = {
      draggable: false,
      id: 'bg',
      name: 'bg',
      image,
      x: 0,
      y: 0,
      offsetX: 0,
      offsetY: 0,
      rotation: 0,
      width: newWidth,
      height: newHeight
    }

    // 记录图片缩放比率和原始宽高
    this.imageScale = scale
    this.imageRealWidth = this.getBitmapDPISize(width, height).real_width
    this.imageRealHeight = this.getBitmapDPISize(width, height).real_height
    this.imageRealScale = this.bgLayerConfig.width / this.imageRealWidth
    // 画布配置
    this.stageConfig = {
      width: stageWidth,
      height: stageHeight
    }

    // --------- 新增initScale计算方式 ---------
    // 参考 dot-matrix-canvas.vue 的 initCanvas
    let containerWidth = this.containerWidth || stageWidth
    let containerHeight = this.containerHeight || stageHeight
    let contentWidth = width
    let contentHeight = height
    // 计算缩放比例，使图片内容正好适配容器
    const scaleW = containerWidth / contentWidth
    const scaleH = containerHeight / contentHeight
    const initScale = Math.min(scaleW, scaleH, 1)
    this.initScale = initScale

    // --------- END ---------

    let stage = this.getStage()
    stage.on('mousedown', e => {
      if (e.target === e.target.getStage() || e.target.attrs.name == 'bg') {
        if (this.debug) {
          console.log('-------stage.mousedown-------')
        }
        if (this.activeTool) {
          console.log(`-------触发工具：${this.activeTool.label}-------`)
          if (get(this, 'activeTool.mousedown')) {
            this.activeTool.mousedown(e, this, this.activeTool)
          }
        }
      }
    })

    stage.on('mousemove', e => {
      if (this.activeTool) {
        if (get(this, 'activeTool.mousemove')) {
          this.activeTool.mousemove(e, this, this.activeTool)
        }
      }
    })
    stage.on('mouseup', e => {
      if (this.debug) {
        console.log('-------stage.mouseup-------')
      }
      if (this.activeTool) {
        if (get(this, 'activeTool.mouseup')) {
          this.activeTool.mouseup(e, this, this.activeTool)
        }
      }
    })
    // 拖拽移动后
    stage.on('dragend', e => {
      if (get(e, 'target.attrs.id')) {
        let index = this.store.findIndex(v => v.id == e.target.attrs.id)
        this.config.getCompThis().$set(this.store, index, {
          ...this.store[index],
          x: e.target.x(),
          y: e.target.y()
        })
        if (this.debug) {
          console.log('-------stage.dragend-------')
        }
        this.saveStoreState()
      }
    })
    stage.on('dblclick', e => {
      if (this.debug) {
        console.log('-------stage.dblclick-------')
      }
      if (this.activeTool) {
        if (get(this, 'activeTool.dblclick')) {
          this.activeTool.dblclick(e, this, this.activeTool)
        }
      }
    })
    // stage.on('dragend', e => {
    //   console.log(e)
    //   this.saveStoreState()
    // })
    // 计算 initScale，参考 dot-matrix-canvas.vue
    // initScale = 背景图实际渲染宽度 / 原始宽度
    // this.initScale = 20
    // if (this.bgLayerConfig && this.bgLayerConfig.image) {
    //   const img = this.bgLayerConfig.image
    //   if (img.width) {
    //     this.initScale = img.width / this.bgLayerConfig.width
    //   }
    // }
    // 渲染点阵和 mark
    this.renderPointsAndMarks()
  }
  // 像素坐标转换成物理坐标
  getBitmapDPISize(imgWidth, imgHeight) {
    const densityDPI = 300 / 25.4 // 300 DPI
    const real_width = imgWidth / densityDPI // 像素转换成物理
    const real_height = imgHeight / densityDPI
    return { real_width, real_height }
  }
  /**
   * 渲染点阵和 mark
   */
  renderPointsAndMarks() {
    const stage = this.getStage()
    // 点阵图层
    let pointsLayer = stage.findOne('#pointsLayer')
    if (!pointsLayer) {
      pointsLayer = new Konva.Layer({ id: 'pointsLayer' })
      stage.add(pointsLayer)
    } else {
      pointsLayer.removeChildren()
    }
    // mark图层
    let marksLayer = stage.findOne('#marksLayer')
    if (!marksLayer) {
      marksLayer = new Konva.Layer({ id: 'marksLayer' })
      stage.add(marksLayer)
    } else {
      marksLayer.removeChildren()
    }
    // 渲染点阵
    if (this.points && this.points.length) {
      const lines = this.processPointsData(this.points)
      lines.forEach(lineData => {
        const line = new Konva.Line({
          points: lineData.points,
          stroke: '#000',
          strokeWidth: lineData.linewidth,
          lineCap: 'round',
          lineJoin: 'round',
          tension: 0
        })
        pointsLayer.add(line)
      })
      pointsLayer.batchDraw()
    }
    // 渲染 mark
    if (this.marks && this.marks.length) {
      const rectangles = this.processMarksData(this.marks)
      rectangles.forEach(rectData => {
        const rect = new Konva.Rect({
          x: rectData.x,
          y: rectData.y,
          width: rectData.width,
          height: rectData.height,
          stroke: 'transparent',
          strokeWidth: 1,
          fill: 'transparent',
          id: 'mark'
        })
        console.log(rect)
        marksLayer.add(rect)
      })
      marksLayer.batchDraw()
      // 渲染完mark后，自动定位到第一个mark
      // if (rectangles.length > 0) {
      //   this.focusMark(rectangles[0])
      // }
    }
  }
  /**
   * 点阵数据处理（分轴缩放，保证点阵和图片完全对齐）
   */
  processPointsData(points) {
    const lines = []
    let currentLine = null

    for (const point of points) {
      const { x, y, linewidth, stroke_start, stroke_end } = point
      if (stroke_start || !currentLine) {
        currentLine = {
          points: [],
          linewidth: linewidth || 2
        }
        lines.push(currentLine)
      }
      if (stroke_end) {
        currentLine = null
        continue
      }
      // 分轴缩放
      console.log(this.imageRealScale, this.initScale)
      const pointX = x * this.imageRealScale
      const pointY = y * this.imageRealScale

      if (currentLine) {
        currentLine.points.push(pointX, pointY)
      }
    }
    return lines
  }
  /**
   * mark数据处理
   */
  processMarksData(marks) {
    // 参考 dot-matrix-canvas.vue 的 processMarksData，使用 this.initScale
    const rectangles = []
    for (const mark of marks) {
      const { topx, topy, bottomx, bottomy } = mark
      if (
        topx === undefined ||
        topy === undefined ||
        bottomx === undefined ||
        bottomy === undefined
      )
        continue
      const x1 = topx * this.initScale
      const y1 = topy * this.initScale
      const x2 = bottomx * this.initScale
      const y2 = bottomy * this.initScale
      const width = Math.abs(x2 - x1)
      const height = Math.abs(y2 - y1)
      const x = Math.min(x1, x2)
      const y = Math.min(y1, y2)
      rectangles.push({ x, y, width, height })
    }
    return rectangles
  }
  /**
   * 聚焦到指定的mark
   * @param {object} markData { x, y, width, height }
   */
  focusMark(markData) {
    if (!markData) return
    const stageWidth = this.stageConfig.width
    const stageHeight = this.stageConfig.height
    const padding = 25

    const stage = this.getStage()
    const bgNode = stage.findOne('#bg')
    const pointsLayer = stage.findOne('#pointsLayer')
    const marksLayer = stage.findOne('#marksLayer')

    // 计算缩放比例
    const scaleX = (stageWidth - 2 * padding) / markData.width
    const scaleY = (stageHeight - 2 * padding) / markData.height
    const newScale = Math.min(scaleX, scaleY)

    // 计算居中偏移
    const markCenterX = markData.x + markData.width / 2
    const markCenterY = markData.y + markData.height / 2
    const newGroupX = stageWidth / 2 - markCenterX * newScale
    const newGroupY = stageHeight / 2 - markCenterY * newScale

    // 应用到所有图层
    if (bgNode) {
      bgNode.position({ x: newGroupX, y: newGroupY })
      bgNode.scale({ x: newScale, y: newScale })
      bgNode.getLayer().batchDraw()
    }
    if (pointsLayer) {
      pointsLayer.position({ x: newGroupX, y: newGroupY })
      pointsLayer.scale({ x: newScale, y: newScale })
      pointsLayer.batchDraw()
    }
    if (marksLayer) {
      marksLayer.position({ x: newGroupX, y: newGroupY })
      marksLayer.scale({ x: newScale, y: newScale })
      marksLayer.batchDraw()
    }
  }
  get pointer() {
    const { x, y } = this.getStage().getPointerPosition()
    return { x, y }
  }
  get stage() {
    return this.config.getCompThis().$refs.stage.getStage()
  }
  getStage() {
    return this.config.getCompThis().$refs.stage.getStage()
  }
  setActiveTool(tool) {
    if (tool) {
      this.activeTool = tool
      if (get(this, 'activeTool.onActive')) {
        this.activeTool.onActive(this)
      }
    } else {
      if (get(this, 'activeTool.onInactive')) {
        this.activeTool.onInactive(this)
      }
      this.activeTool = null
    }
  }
  /**
   * 设置背景图层旋转角度并更新宽高和偏移
   * @param {number} rotate 旋转角度，必须是90的倍数
   */
  async setBgLayerRotate(rotate) {
    // 验证旋转角度是否为90的倍数
    if (rotate % 90 !== 0) {
      console.error('旋转角度必须是90的倍数')
      return
    }
    const rotation = Math.abs(rotate % 360)

    // 旋转点的函数
    const rotatePoint = ({ x, y }, rad) => {
      const rcos = Math.cos(rad)
      const rsin = Math.sin(rad)
      return { x: x * rcos - y * rsin, y: y * rcos + x * rsin }
    }

    // 在中心点旋转的函数
    function rotateAroundCenter(node, rotation) {
      // 当前旋转原点（0, 0）相对于所需原点 - 中心（node.width()/2, node.height()/2）
      const topLeft = { x: -node.width() / 2, y: -node.height() / 2 }

      // 当前旋转角度
      const current = rotatePoint(topLeft, Konva.getAngle(node.rotation()))

      // 新旋转角度
      const rotated = rotatePoint(topLeft, Konva.getAngle(rotation))

      // 计算偏移量
      const dx = rotated.x - current.x
      const dy = rotated.y - current.y
      return {
        rotation,
        dx,
        dy
      }
    }

    let { dx, dy } = rotateAroundCenter(this.stage.findOne('#bg'), rotation)
    this.bgLayerConfig.rotation = rotation
    this.bgLayerConfig.x += dx // 更新 x 坐标
    this.bgLayerConfig.y += dy // 更新 y 坐标

    if (this.debug) {
      console.log(
        `背景图层旋转角度设置为: ${rotate}°，新位置: (${this.bgLayerConfig.x}, ${this.bgLayerConfig.y})，新宽高: (${this.bgLayerConfig.width}, ${this.bgLayerConfig.height})`
      )
    }
  }

  //----------历史记录-------------S-//
  saveStoreState() {
    if (this.debug) {
      console.log('-----历史记录已保存----')
    }
    this.history.push(cloneDeep(this.store))
  }
  // redo() {}
  undo() {
    if (this.history.length > 2) {
      let store = cloneDeep(this.history[this.history.length - 2]) || []
      this.store = store.map(v => {
        return {
          ...v,
          image: ''
        }
      })
      this.store.forEach(item => {
        if (item.imageUrl) {
          loadImage(imageList[item.imageUrl]).then(({ image }) => {
            let index = this.store.findIndex(v => v.id == item.id)
            this.config.getCompThis().$set(this.store, index, {
              ...this.store[index],
              image: image
            })
          })
        }
      })
      this.history.pop()
    } else {
      if (this.debug) {
        console.log('没有更多的历史记录可以撤销')
      }
    }
  }
  //----------历史记录-------------E-//
}

export async function createApp(config) {
  const instance = new KonvaCanvas(config)
  await instance.init()
  return instance
}

export function loadImage(src) {
  return new Promise((resolve, reject) => {
    const image = new window.Image()
    if (src.includes(';base64')) {
      image.src = src // 直接使用 base64 字符串
    } else {
      image.src = src + '?time=' + new Date().valueOf()
      image.crossOrigin = 'Anonymous'
    }

    image.onload = () => {
      resolve({
        image: image,
        width: image.width,
        height: image.height
      })
    }
    image.onerror = () => {
      console.error('底图加载失败')
      reject('file load iamge')
    }
  })
}

export const createTextarea = (app, textNode, onUpdated) => {
  textNode.hide()

  const textPosition = textNode.getClientRect()
  const areaPosition = {
    x: app.stage.container().offsetLeft + textPosition.x,
    y: app.stage.container().offsetTop + textPosition.y
  }

  const textarea = document.createElement('textarea')
  app.stage.container().appendChild(textarea)

  const transform = `translateY(-${Math.round(textNode.fontSize() / 20)}px)`

  Object.assign(textarea.style, {
    position: 'absolute',
    display: 'inline-block',
    minHeight: '1em',
    backfaceVisibility: 'hidden',
    resize: 'none',
    background: 'transparent',
    overflow: 'hidden',
    overflowWrap: 'break-word',
    boxSizing: 'content-box',
    top: `${areaPosition.y}px`,
    left: `${areaPosition.x}px`,
    width: `${
      textNode.width() * textNode.scaleX() - textNode.padding() * 2 + 10
    }px`,
    height: `${
      textNode.height() * textNode.scaleX() - textNode.padding() * 2 + 5
    }px`,
    fontSize: `${textNode.fontSize() * textNode.scaleX()}px`,
    border: 0,
    padding: 0,
    margin: 0,
    outline: 0,
    lineHeight: textNode.lineHeight().toString(),
    fontFamily: textNode.fontFamily(),
    textAlign: textNode.align(),
    color: textNode.fill(),
    caretColor: textNode.fill(),
    zIndex: '99999',
    transformOrigin: 'left top',
    transform: transform
  })

  textarea.value = textNode.text()
  textarea.style.height = `${textarea.scrollHeight + 3}px`
  textarea.focus()

  function removeTextarea() {
    textarea.parentNode?.removeChild(textarea)
    window.removeEventListener('click', handleOutsideClick)
    textNode.show()
  }

  textarea.addEventListener('keydown', e => {
    e.stopPropagation()
    if (e.key === 'Escape') {
      removeTextarea()
    }
  })

  textarea.addEventListener('input', () => {
    const text = textarea.value.replace(/\n/g, '<br/>')
    const tempDiv = document.createElement('div')
    tempDiv.innerHTML = text
    tempDiv.style.position = 'absolute'
    tempDiv.style.visibility = 'hidden'
    tempDiv.style.whiteSpace = 'pre-wrap'
    document.body.appendChild(tempDiv)

    const newWidth = tempDiv.offsetWidth + 10
    textarea.style.width = `${newWidth * textNode.scaleX()}px`
    document.body.removeChild(tempDiv)

    const lineHeight = parseFloat(getComputedStyle(textarea).lineHeight)
    const rows = textarea.value.split('\n').length
    textarea.style.height = 'auto'
    const newHeight = lineHeight * rows
    textarea.style.height = `${newHeight}px`
    textNode.width(newWidth)
    textNode.height(newHeight / textNode.scaleY())
  })

  function handleOutsideClick(e) {
    if (e.target !== textarea) {
      if (textarea.value.trim().length >= 1) {
        textNode.text(textarea.value)
      } else {
        app.remove(textNode)
      }
      removeTextarea()
      onUpdated()
    }
  }

  setTimeout(() => {
    window.addEventListener('click', handleOutsideClick)
  })
}
