# Leafer.js 教学批注系统 Demo




## 结构

```
src/
├── api/                    # Mock 数据文件
│   ├── marklist.json      # 标记区域数据
│   ├── pageinfo.json      # 页面基础信息
│   ├── point.json         # 笔画数据点
│   └── page-point.json    # 页面笔画数据
├── pages/leafer/          # Leafer.js 相关组件
│   ├── index.vue          # 主页面组件
│   ├── leafer.js          # Leafer 核心逻辑
│   └── leafer-util.js     # 工具函数
└── main.js                # 应用入口
```

## Mock 数据说明

### 1. 页面信息 (pageinfo.json)

包含页面的基础信息：

```json
{
  "id": "页面唯一标识",
  "bookName": "教材名称",
  "pageID": 80261,
  "pageNumber": 1,
  "pageTotal": 14,
  "url": "页面图片URL地址",
  "markCount": 45
}
```

### 2. 标记区域数据 (marklist.json)

定义页面上的矩形标记区域，用于标注题目区域、答案区域等：

```json
{
  "id": "标记唯一标识",
  "top": { "x": 1831.49, "y": 365.62 },    // 左上角坐标
  "bottom": { "x": 1968.24, "y": 396.70 }, // 右下角坐标
  "questionID": 0,                          // 题目ID
  "childQuestionID": "ai-practice",         // 子题目类型
  "pageID": 80261,                          // 页面ID
  "createTime": 1751254686                  // 创建时间戳
}
```

**常见标记类型：**
- `ai-practice` - AI 练习
- `kp-analyze` - 知识点分析
- `q-dictation` - 题目听写
- `ai-grade` - AI 评分
- `q-analyze` - 题目分析
- `q-collect` - 题目收藏

### 3. 笔画数据 (point.json / page-point.json)

存储手写笔画的轨迹数据，每个笔画包含一系列坐标点和笔画属性：

```json
{
  "pageId": 82412,
  "strokes": "x1,y1,pressure1,isStart,isEnd,timestamp1;x2,y2,pressure2,isStart,isEnd,timestamp2;...",
  "isSaved": false,
  "isDraw": false
}
```

**笔画数据格式说明：**
- `x, y` - 坐标位置
- `pressure` - 压感值 (0-1)
- `isStart` - 是否为笔画起点
- `isEnd` - 是否为笔画终点
- `timestamp` - 时间戳

## 使用示例

### 基本用法

```vue
<template>
  <div>
    <div class="leafer-container" id="leafer-container"></div>
  </div>
</template>

<script>
import { createLeaferAnnotate } from './leafer'
import pageinfo from '../../api/pageinfo.json'
import markList from '../../api/marklist.json'
import pagePoints from '../../api/page-point.json'

export default {
  async mounted() {
    this.instance = await createLeaferAnnotate({
      view: 'leafer-container',
      pageUrl: pageinfo.url,
      marks: markList,
      points: pagePoints,
    })
  }
}
</script>
```

### 初始化 Leafer 实例

```javascript
import { createLeaferAnnotate } from './leafer'

const instance = await createLeaferAnnotate({
  view: 'container-id',           // 容器元素ID
  pageUrl: 'image-url',           // 页面图片URL
  marks: [...],                   // 标记区域数组
  points: [...]                   // 笔画数据数组
})
```

## 核心功能实现

### 1. 图片加载与渲染

```javascript
// 加载图片并获取尺寸信息
let { url, width, height } = await loadImage(pageUrl)

// 创建页面框架
this.pageFrame = new Frame({
  width: width,
  height: height,
})

// 添加图片到框架
this.pageFrame.add(new Image({ url: url }, width, height))
```

### 2. 标记区域渲染

```javascript
// 根据坐标数据创建矩形标记
const rect = new Rect({
  x: left,
  y: top,
  width: right - left,
  height: bottom - top,
  fill: 'transparent',
  stroke: '#ff0000',
  strokeWidth: 1
})
```

### 3. 手写笔画渲染

```javascript
// 将笔画数据转换为 SVG 路径
const path = optimizedCreatePathsFromStrokes(strokesStr, scaleX, scaleY)

// 创建路径元素
const pathElement = new Path({
  path: path,
  stroke: '#000000',
  strokeWidth: 3,
  fill: null,
  strokeCap: 'round',
  strokeJoin: 'round'
})
```

## 安装与运行

### 环境要求

- Node.js >= 14.0.0
- npm 或 yarn 或 pnpm

### 安装依赖

```bash
npm install
```

### 启动开发服务器

```bash
npm run serve
```

### 构建生产版本

```bash
npm run build
```

## 相关资源

- [Leafer.js 官方文档](https://www.leaferjs.com/)

## 许可证

本项目仅供学习和参考使用。